
/* Copyright (C) 2001-2007 Monotype Imaging Inc. All rights reserved. */

/* Confidential information of Monotype Imaging Inc. */

/* fs_bitio.c */


#include "fs_itype.h"

#if defined(FS_ACT3) || defined (FS_CCC)


/****************************************************************/
/*
* This performance of this next routine is vital to the runtime
* performance of ACT. If speed were not an issue we could use the
* following code:
*
*    FS_ULONG MTX_BITIO_ReadValue( BITIO *t, FS_LONG numberOfBits )
*        {
*        FS_BYTE *array = t->array;
*        FS_LONG index = t->index;
*        FS_ULONG r = 0;
*
*        while (numberOfBits--)
*            {
*            r <<= 1;
*            if (array[index >> 3] & mask[index & 7])
*                r |= 1;
*            index++;
*            }
*        t->index = index;
*        return r;
*        }
*
*
* But speed is an issue and on my box (PentiumII) the real code
* which follows is 3x faster in decoding an average file.
*
* It would be well worthwhile to have an assembler geek spend a
* few days optimizing this code -- or changing the algorithm to
* take advantage of specialized instructions like the 'bitfield'
* instructions of the motorola 680X0 family.
*
*/

static FS_CONST FS_ULONG mask32[] =
{
    0x00000000,
    0x00000001, 0x00000003, 0x00000007, 0x0000000F,
    0x0000001F, 0x0000003F, 0x0000007F, 0x000000FF,
    0x000001FF, 0x000003FF, 0x000007FF, 0x00000FFF,
    0x00001FFF, 0x00003FFF, 0x00007FFF, 0x0000FFFF,
    0x0001FFFF, 0x0003FFFF, 0x0007FFFF, 0x000FFFFF,
    0x001FFFFF, 0x003FFFFF, 0x007FFFFF, 0x00FFFFFF,
    0x01FFFFFF, 0x03FFFFFF, 0x07FFFFFF, 0x0FFFFFFF,
    0x1FFFFFFF, 0x3FFFFFFF, 0x7FFFFFFF, 0xFFFFFFFF
};

/****************************************************************/
/* get the next <numberOfBits> bits from the bitarray <t> */
FS_ULONG MTX_BITIO_ReadValue( BITIO *t, FS_LONG numberOfBits )
{
    FS_ULONG first, last, r, mask;
    FS_BYTE *p;
    int n, shift;

    if ( !numberOfBits )
        return 0;
    first = t->index >> 3;
    r = t->index + numberOfBits - 1;
    last = r >> 3;
    n = 1 + last - first;

    shift = 7 - (r & 7);
    mask = mask32[numberOfBits];
    p = t->array + first;

    switch (n)
    {
    case 1:
        r = *p;
        r >>= shift;
        r &= mask;
        break;
    case 2:
        r = *p++;
        r = (r << 8) | *p;
        r >>= shift;
        r &= mask;
        break;
    case 3:
        r = *p++;
        r = (r << 8) | *p++;
        r = (r << 8) | *p;
        r >>= shift;
        r &= mask;
        break;
    case 4:
        r = *p++;
        r = (r << 8) | *p++;
        r = (r << 8) | *p++;
        r = (r << 8) | *p;
        r >>= shift;
        r &= mask;
        break;
    default:
        /* We can never span more than 5 bytes with <= 32 bits, but
        * we can span 5 bytes with as few as 26 requested bits, eg:
        * [.......x][xxxxxxxx][xxxxxxxx][xxxxxxxx][x.......] */
#if SIZEOF_LONG >= 5
        /* which is no problem on machines with longs of 40+ bits */
        {
            unsigned long temp;

            temp = *p++;
            temp = (temp << 8) | *p++;
            temp = (temp << 8) | *p++;
            temp = (temp << 8) | *p++;
            temp = (temp << 8) | *p;
            temp >>= shift;
            r = (FS_ULONG)temp & mask;
        }
#else
        /* which is rather ugly on a 32 bit machine ... fortunately
        * spans of 5 are rather uncommon ... only 28 of 256 possible
        * fetches span 5 bytes.  Moreover, they all occur for tokens
        * with very low frequency - a 26 bit code spanning 5 bytes
        * will occur once in every 2**29 fetches; a 32 bit code which
        * spans 5 bytes will occur once in every 7*2**35 fetches.
        *
        * By the way: a 32 bit code would signify a token which had a
        * frequency of 1 in 2**32 in the data ,,, but which occurred
        * at least twice (in order to be chosen a token) and since the
        * minimum token size is 2 bytes, this will occur only for files
        * which exceed 8 gigabytes uncompressed.
        *
        * Like I said ... rather uncommon.
        */
        first = numberOfBits / 2;
        last = numberOfBits - first;
        r = MTX_BITIO_ReadValue(t, first);
        r = (r << last) | MTX_BITIO_ReadValue(t, last);
#endif /* SIZEOF_LONG >= 5  */

        break;
    }

    t->index += numberOfBits;
    return r;
}

/* the file header is byte aligned, hence the special cases */
/****************************************************************/
FS_ULONG MTX_BITIO_Read8( BITIO *t )
{
    FS_ULONG r;
    FS_ULONG index = t->index;
    FS_BYTE *array = t->array;

    if (0 == (index & 7))
    {
        r = array[index >> 3];
        t->index += 8;
    }
    else
        r = MTX_BITIO_ReadValue(t, 8);
    return r;
}

/****************************************************************/
FS_ULONG MTX_BITIO_Read32( BITIO *t )
{
    FS_ULONG r;
    FS_ULONG index = t->index;
    FS_BYTE *array = t->array;

    if (0 == (index & 7))
    {
        index >>= 3;
        r = array[index++];
        r = (r << 8) | array[index++];
        r = (r << 8) | array[index++];
        r = (r << 8) | array[index];
        t->index += 32;
    }
    else
        r = MTX_BITIO_ReadValue(t, 32);
    return r;
}

/****************************************************************/
/* it's easier (and more efficient) to make a local variable */
BITIO *MTX_BITIO_Create( _DS_ FS_VOID* memPtr, FS_LONG memSize )
{
    BITIO *r;

#ifdef FS_MEM_DBG
    STATE.memdbgid = "BITIO";
#endif
    r = (BITIO *)FSS_malloc(_PS_ sizeof(BITIO));
    if (!r)
        return NULL;

    r->array = (FS_BYTE *)memPtr;
    r->index = 0;
    r->max = memSize << 3;

    return r;
}


/****************************************************************/
#endif /* FS_ACT3 */
